/**************************************************************************************
 
   Copyright (c) Hilscher GmbH. All Rights Reserved.
 
 **************************************************************************************
 
   Filename:
    $Workfile: $
   Last Modification:
    $Author: Robert $
    $Modtime: $
    $Revision: 12212 $
   
   Targets:
     rcX   : yes
 
   Description:
     Hook Uart in/out to StdIO routines
       
   Changes:
 
     Version   Date        Author   Description
     ----------------------------------------------------------------------------------
     1         02.03.2011  SS       Initial version
     
**************************************************************************************/

#include "libc_file_support.h"
#include "rX_Includes.h"

static RX_HANDLE          s_hUart;
static RX_URT_HANDLER_UN  s_unUartHandler;
static RX_HANDLE          s_ptIntUart;
static RX_HANDLE          s_hRecvSem;
static CHAR               s_bRecv;

/*****************************************************************************/
/*! rcX Function called when a new character is available at UART
*   \param hUart   UART Handle
*   \param pvUser  UART connector internal data
*   \param uRxChar Received char                                             */
/*****************************************************************************/
static void UartRxReady(RX_HANDLE hUart, void* pvUser, UINT uRxChar)
{
  s_bRecv = uRxChar;
  (void)rX_SemPutSemaphore(s_hRecvSem);
}

/*****************************************************************************/   
/*! rcX Function called to write a byte stream to the UART                    
*   \param pvBuffer                                               
*   \param ulSize                                      
*   \param pvUser                                                            */   
/*****************************************************************************/ 
static void UartWrite(const void* pvBuffer, unsigned long ulSize, void *pvUser)
{
  unsigned char* pbData = (unsigned char*)pvBuffer;
   
  while(ulSize-- > 0)
  {
    while (RX_OK != Drv_UrtSendCharacter(s_hUart, *pbData));
    pbData++;
  }
}

/*****************************************************************************/   
/*! rcX Function called to read a byte from the UART                    
*   \param pvBuffer                                               
*   \param ulSize                                      
*   \param pvUser                                                            */    
/*****************************************************************************/ 
static int UartRead(void* pvBuffer, unsigned long* pulSize, void* pvUser)
{
  *pulSize = 1;
  (void)rX_SemWaitForSemaphore(s_hRecvSem, RX_INFINITE);
  (void)Drv_UrtSendCharacter(s_hUart,s_bRecv);
  *(CHAR*)pvBuffer = s_bRecv;
  return 1; 
}

/*****************************************************************************/   
/*! rcX Function called to init the UART
 *  \param szUart      Name of the UART instance
 *  \param szInterrupt Name of the UART's interrupt instance
 *  \return RX_OK on success                                                 */
/*****************************************************************************/ 
RX_RESULT UartStdIOHookInit(const char* szUart, const char* szInterrupt)
{
  RX_RESULT erXRes = RX_OK;
  
  if ( (RX_OK != (erXRes = rX_MemAllocateMemory(&s_hRecvSem, RX_SEMAPHORE_SIZE))) ||
       (RX_OK != (erXRes = rX_SemCreateSemaphore(NULL, s_hRecvSem, 0))) )
  {
    (void)rX_MemFreeMemory(s_hRecvSem);
     
  } else 
  {
    if( (RX_OK == (erXRes = Drv_UrtIdentifyUart(szUart, 0, &s_hUart))) &&
        (RX_OK == (erXRes = Drv_UrtInitializeUart(s_hUart, UartRxReady, NULL, NULL, NULL, TRUE, &s_unUartHandler))) &&
        (RX_OK == (erXRes = Drv_UrtEnableUart(s_hUart))) &&
        (RX_OK == (erXRes = Drv_UrtEnableReceiver(s_hUart, TRUE))) )
    {
      if(RX_OK == (erXRes = Drv_IntIdentifyInterrupt(szInterrupt, 0, &s_ptIntUart)))
      { 
        (void)Drv_IntInitializeInterrupt(s_ptIntUart,
                                         s_unUartHandler.fnIrq,
                                         s_hUart);
        (void)Drv_IntEnableInterrupt(s_ptIntUart);
      }  
  
      libc_file_init();
      
      /* init the libc part */
      Libc_RegisterStdOutFileHook(0, UartWrite, 0);
      Libc_RegisterStdOutFileHook(1, UartWrite, 0);  
      Libc_RegisterStdInFileHook(UartRead, 0);
    } else
    {
      (void)rX_SemDeleteSemaphore(s_hRecvSem);
      (void)rX_MemFreeMemory(s_hRecvSem);
    }
  }
  
  return erXRes;
}
